/* ****************************************************************************
 * Copyright: 2017-2025 RAYLASE GmbH
 * This source code is the proprietary confidential property of RAYLASE GmbH.
 * Reproduction, publication, or any form of distribution to
 * any party other than the licensee is strictly prohibited.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#pragma once

#include <optional>
#include <stdexcept>
#include <string>

class RpcRemoteException : public std::runtime_error
{
private:
	std::optional<std::exception> _Inner;
	std::optional<std::string> _StackTrace;
	std::string _what;
	void init_what();
	virtual const char* base_what() const noexcept;

public:
	RpcRemoteException()
	    : runtime_error("Rpc remote exception occurred")
	{
		init_what();
	}
	RpcRemoteException(const std::string& error)
	    : runtime_error(error)
	{
		init_what();
	}
	RpcRemoteException(const std::string& error, const std::exception& inner)
	    : runtime_error(error),
	      _Inner(inner)
	{
		init_what();
	}
	RpcRemoteException(const std::string& error, const std::string& stackTrace)
	    : runtime_error(error),
	      _StackTrace(stackTrace)
	{
		init_what();
	}
	RpcRemoteException(const std::string& error, const std::exception& inner, const std::string& stackTrace)
	    : runtime_error(error),
	      _Inner(inner),
	      _StackTrace(stackTrace)
	{
		init_what();
	}
	RpcRemoteException(const RpcRemoteException& other)
	    : runtime_error(other.base_what()),
	      _Inner(other._Inner),
	      _StackTrace(other._StackTrace)
	{
		init_what();
	}

	const std::optional<std::exception>& Inner() const { return _Inner; }
	const std::optional<std::string>& StackTrace() const { return _StackTrace; }

	// jws: this explicit declaration prevents the compiler from *implicitly* declaring the operator=(), which results in a very large number of deprecation
	// warnings being emitted:
	//  "warning: implicitly-declared RpcRemoteException& RpcRemoteException::operator=(const RpcRemoteException&) is deprecated [-Wdeprecated-copy]"
	RpcRemoteException& operator=(const RpcRemoteException&) = default;

	virtual const char* what() const noexcept;

	inline std::string ToString() const { return std::string("RpcRemoteException: ") + what(); }
};
